
      subroutine com6(iou, istn, iyr, idoy, sat, wave, 
     .                ik, icm, ierr)
      
c        Returns centre of mass value for requested station, date, satellite,
c        and laser wavelength. Units: [mm].

c        This subroutine will return all the CoM entries matching ALL the 
c        requested parameters, ignoring all the entries that fail to 
c        match ANY of the requested parameters. An error will be generated 
c        for epochs outside the tabulated data or unknown combinations of 
c        epochs and wavelengths.

c        The behaviour of the subroutine under error conditions is controlled 
c        by the logical variable RETURN_DEFAULT. If set to FALSE, the returned 
c        CoM values are zero when an error in encountered. If set to TRUE, it 
c        returns default values computed from averaging the CoM tables.

c        At the moment there is no clean way to choose between coexisting 
c        system configurations (e.g. 7840 between 2007 and 2015). Although 
c        the differences are minimal, the CoM entries are arranged so that 
c        the last output returned will be the correct choice most of the time.

c        Input:
c        ------
c        iou  : unit number for reading the data files
c        istn : station CDP number, eg. 7810
c        iyr  : date in integer years, e.g., 2009
c        idoy : day of year, e.g. 354
c        sat  : satellite code
c               l : LAGEOS
c               m : LAGEOS-2
c               e : Etalon-1/2
c               a : Ajisai
c               s : LARES
c               r/t : Starlette/Stella    (indifferent)
c               u : LARES-2
c        wave : laser wavelength (nm)

c        Output:
c        -------
c        ik   : number of CoM values being returned. Used to extract last
c               entry from icm array of possible values
c        icm  : array of possible CoM corrections [mm]
c        ierr : error code
c               0 : success
c               1 : error (a value of zero is returned for the CoM)
c               2 : error. Default CoM value returned. This behaviour is 
c                    achieved setting the variable RETURN_DEFAULT to True.

c        Author: J. Rodríguez (jcrodriguez@mitma.es).

c        Reference:
c        Rodriguez J., Otsubo T., Appleby G. Upgraded Modelling for the 
c        Determination of Centre of Mass Corrections of Geodetic SLR 
c        Satellites: Impact on Key Parameters of the Terrestrial Reference
c        Frame. Journal of Geodesy, 2019. doi: 10.1007/s00190-019-01315-0

c        last changed: JR 231128

         implicit none
         character sat*1
         character*80 fn_lg1, fn_lg2, fn_et1, fn_aji, fn_las, fn_str,
     .                fn_la2, COM_PATH    
         integer iou, istn, iyr, idoy, wave, ik, ierr, imjd, imode
         double precision rmjdz, icm(20), rms
         double precision com_lg1(1000,7), com_lg2(1000,7), 
     .                    com_eta(1000,7), com_aji(1000,7), 
     .                    com_las(1000,7), com_str(1000,7),
     .                    com_la2(1000,7)


c        Set RETURN_DEFAULT to True to let com6 handle default CoM values
c        N.B. this may mask all errors: the caller should check the output 
c        error code in any case.

         logical :: RETURN_DEFAULT=.TRUE.

         data imode/0/
         save
              
         if (imode .eq. 0) then        ! first entry, read tables
            COM_PATH = 'com/'          ! change path to CoM tables here
            fn_lg1 = trim(COM_PATH) // 'com_lg1.dat'
            fn_lg2 = trim(COM_PATH) // 'com_lg2.dat'
            fn_et1 = trim(COM_PATH) // 'com_et1.dat'
            fn_aji = trim(COM_PATH) // 'com_aji.dat'
            fn_las = trim(COM_PATH) // 'com_las.dat'
            fn_str = trim(COM_PATH) // 'com_str.dat'
            fn_la2 = trim(COM_PATH) // 'com_la2.dat'

            call com_table_read(iou, fn_lg1, com_lg1)
            call com_table_read(iou, fn_lg2, com_lg2)
            call com_table_read(iou, fn_et1, com_eta)
            call com_table_read(iou, fn_aji, com_aji)
            call com_table_read(iou, fn_las, com_las)
            call com_table_read(iou, fn_str, com_str)
            call com_table_read(iou, fn_la2, com_la2)
            imode=1
         endif
       
c        Normal entry to get appropriate CoM

         ierr = 1

         call modjd_local(iyr-1900, 1, 1, 0, 0, rmjdz)
         imjd = idint(rmjdz) + idoy - 1

         
         select case (sat)
            case ('l')      ! LAGEOS
               call com_get(istn, imjd, wave, com_lg1, ik, icm, ierr)
            case ('m')      ! LAGEOS-2
               call com_get(istn, imjd, wave, com_lg2, ik, icm, ierr)
            case ('e')      ! Etalon
               call com_get(istn, imjd, wave, com_eta, ik, icm, ierr)
            case ('a')      ! Ajisai
               call com_get(istn, imjd, wave, com_aji, ik, icm, ierr)
            case ('s')      ! LARES
               call com_get(istn, imjd, wave, com_las, ik, icm, ierr)
            case ('r')      ! Starlette
               call com_get(istn, imjd, wave, com_str, ik, icm, ierr)
            case ('t')      ! Stella
               call com_get(istn, imjd, wave, com_str, ik, icm, ierr)
            case ('u')      ! LARES-2
               call com_get(istn, imjd, wave, com_la2, ik, icm, ierr)
            case default
               print *, ' Unknown satellite code [com6] '
               return
         end select

c        If the variable RETURN_DEFAULT is set to True, check ierr and, if 1 
c        (error in retrieving a suitable value), get the default CoM for the 
c        satellite. In this case set ierr to 2 to indicate that an error 
c        was found and that the default CoM value was used.

         if ((ierr == 1) .and. (RETURN_DEFAULT .eqv. .True.)) then
            call default_com(sat, icm(1))
            ik = 1
            ierr = 2
         endif         
         
      end subroutine

      
      subroutine com_get(istn, imjd, wave, vcom, ik, icm, ierr)
      
c        Trawls CoM array looking for matching entries 
c        according to station id, epoch, and wavelength

c        Input:
c        istn : station CDP number, eg. 7810
c        imjd : epoch in MJD
c        wave : laser wavelength (nm)
c        vcom : array of CoM data (istn, MJD1, MJD2, wavelength, CoM),
c               read with subroutine com_table_read()

c        Output:
c        ik : number of CoM values being returned <=20
c        icm : array of possible CoM corrections (mm)
c        ierr: success (0) or failure (1) to get a CoM value

         implicit none
         integer istn, imjd, wave, ik, k, ierr
         double precision vcom(1000,5)    ! input 
         double precision icm(20)
         
c        Deal with some possible roundings of the wavelength value;
c        the rest will be catched in the matching condition below
         if ((wave .ge. 600) .and. (wave .le. 700)) wave = 694
         if (wave .ge. 1000) wave = 1064
         
         ierr = 1
         do k = 1,20
            icm(k) = 0
         enddo
         ik = 0
         do k = 1,1000
            if (vcom(k,1) .gt. istn) exit
            if (istn .eq. vcom(k,1)) then
c              test for matching wavelength: |requested - table| <= 50 nm
               if (abs(wave - vcom(k,4)) .gt. 50) cycle
               if (imjd .ge. vcom(k,2) .and. imjd .le. vcom(k,3)) then
                  ik = ik + 1
                  icm(ik) = vcom(k,5)
                  ierr = 0
               endif
            endif
         enddo
         
      end subroutine

      
      subroutine com_table_read(iou, fname, vcom)
      
c        Reads CoM table values from specified file onto vcom array

c        Input:
c        iou : unit number for reading the data files
c        fname : file name of CoM table to read

c        Output:
c        vcom : array of CoM data (istn, MJD1, MJD2, wavelength, CoM)

         implicit none
         character(len=*) fname
         character(len=80) :: line
         integer i, iou, istn, idays, ims, iyrs, idaye, ime, iyre, wave
         double precision mjds, mjde, cm
         double precision vcom(1000,7)
         
         open(unit=iou, file=fname, status='old')
         
         do                     ! skip header
            read(iou, '(a80)') line
            if (line(1:1) .eq. '*') then
                if (line(1:13) .eq. '* end_of_head') exit
            else
                rewind(iou)     ! no header in file
                exit
            endif
         enddo
         
c        1181 01 07 1981 01 09 1991  694  253.1
         i=0
         do 
            read(iou,*, end=99) istn, idays, ims, iyrs, 
     .                          idaye, ime, iyre, wave, cm
            call modjd_local(iyrs-1900, ims, idays, 0, 0, mjds)
            call modjd_local(iyre-1900, ime, idaye, 0, 0, mjde)
            i = i+1
            vcom(i,1) = istn
            vcom(i,2) = idint(mjds)
            vcom(i,3) = idint(mjde)
            vcom(i,4) = wave
            vcom(i,5) = cm
         enddo
 99      close(unit=iou)
  
      end subroutine

      
      subroutine com_table_read_rms(iou, fname, vcom)
      
c        Reads CoM table values from specified file onto vcom array

c        Input:
c        iou : unit number for reading the data files
c        fname : file name of CoM table to read

c        Output:
c        vcom : array of CoM data (istn, MJD1, MJD2, wavelength, CoM)

         implicit none
         character(len=*) fname
         character(len=80) :: line
         integer i, iou, istn, idays, ims, iyrs, idaye, ime, iyre, wave
         double precision mjds, mjde, cm, slope, rms
         double precision vcom(1000,7)
         
         open(unit=iou, file=fname, status='old')
         
         do                     ! skip header
            read(iou, '(a80)') line
            if (line(1:1) .eq. '*') then
                if (line(1:13) .eq. '* end_of_head') exit
            else
                rewind(iou)     ! no header in file
                exit
            endif
         enddo
         
c        1181 01 07 1981 01 09 1991  694  253.1
         i=0
         do 
            read(iou,*, end=99) istn, idays, ims, iyrs, 
     .                          idaye, ime, iyre, wave, cm, slope, rms
            call modjd_local(iyrs-1900, ims, idays, 0, 0, mjds)
            call modjd_local(iyre-1900, ime, idaye, 0, 0, mjde)
            i = i+1
            vcom(i,1) = istn
            vcom(i,2) = idint(mjds)
            vcom(i,3) = idint(mjde)
            vcom(i,4) = wave
            vcom(i,5) = cm
            vcom(i,6) = slope
            vcom(i,7) = rms
         enddo
 99      close(unit=iou)
  
      end subroutine
      
      
      subroutine com_table_version(iou, sat, version)
         
c        Reads date string from CoM table for specified satellite

c        Input:
c        iou : unit number for reading the data file
c        sat  : satellite code
c               l : LAGEOS
c               m : LAGEOS-2
c               e : Etalon-1/2
c               a : Ajisai
c               s : LARES
c               r/t : Starlette/Stella    (indifferent)
c               u : LARES-2
c        Output:
c        version : CoM table generation date from header

         implicit none
         integer iou
         character*1 sat
         character*11 version
         character(len=80) :: line, fname, COM_PATH='com/'
         
         select case (sat)
            case ('l')
                fname = trim(COM_PATH) // 'com_lg1.dat'
            case ('m')
                fname = trim(COM_PATH) // 'com_lg2.dat'
            case ('e')
                fname = trim(COM_PATH) // 'com_et1.dat'
            case ('a')
                fname = trim(COM_PATH) // 'com_aji.dat'
            case ('s')
                fname = trim(COM_PATH) // 'com_las.dat'
            case ('r')
                fname = trim(COM_PATH) // 'com_str.dat'
            case ('t')
                fname = trim(COM_PATH) // 'com_str.dat'
            case ('u')
                fname = trim(COM_PATH) // 'com_la2.dat'
            case default
                print *, ' Unknown satellite to com_table_version()'
                return
         end select
    
         open(unit=iou, file=fname, status='old')
         version = '9999-99-99'
         do                     
            read(iou, '(a80)') line
            if (line(1:1) .eq. '*') then
                if (line(1:18) .eq. '* Generation date:') then
                    version = trim(line(19:30))
                    exit
                endif
            else
                print *, ' No header in CoM table'
                exit
            endif
         enddo
         close(iou)
      end subroutine
         
         
      subroutine default_com(sat, cm)
      
c        Returns default CoM value for given satellite

c        Input:
c        sat : satellite code

c        Output:
c        cm : default centre of mass correction
   
         implicit none
         character*1 sat
         double precision cm
         
         select case (sat)
            case ('l')
                cm = 245.4d0
            case ('m')
                cm = 244.8d0
            case ('e')
                cm = 572.9d0
            case ('a')
                cm = 986.2d0
            case ('s')
                cm = 130.2d0
            case ('r')
                cm = 76.1d0
            case ('t')
                cm = 76.1d0
            case ('u')
                cm = 173.5d0
            case default
                print *, ' Unknown satellite to default_com()'
                return
         end select

      end subroutine
      
         
      subroutine modjd_local(iy, imo, iday, ih, im, dmjd)
      
c        Epoch in year, month, day, hour, minute to MJD
         ! local copy for distribution, renamed to avoid clashes with library
         implicit none
         integer iy, imo, iday, ih, im, l, amjd
         double precision dmjd
         save

         l = (imo - 14) / 12
         amjd = aint(365.25 * float(iy + l)) 
     .          + (15048 + iday + 367 * (imo - 2 - 12 * l) / 12
     .          + (1202 - 12 * iy - imo) / 1200)
     
         dmjd = dble(amjd) + (dble(float(ih)) / 24.0d0)
     .          + (dble(float(im)) / 1440.D0)
         return
         
      end subroutine
      
